import streamlit as st
import yfinance as yf
import pandas as pd
import plotly.express as px

from sumy.parsers.plaintext import PlaintextParser
from sumy.nlp.tokenizers import Tokenizer
from sumy.summarizers.lex_rank import LexRankSummarizer

import nltk
nltk.download('punkt')   # 👈 add this line

# --- Text Summarizer Section ---
if st.session_state.show_summarizer:
    st.title("📝 Paste the news in Text Summarizer")

    text = st.text_area("Paste stock-related text/article here:", key="input_text")

    if st.button("Summarize", key="summarize_button"):
        if text:
            parser = PlaintextParser.from_string(text, Tokenizer("english"))
            summarizer = LexRankSummarizer()
            summary = summarizer(parser.document, 3)  # 3 sentences

            st.subheader("Summary")
            for sentence in summary:
                st.write(str(sentence))
        else:
            st.warning("Please enter some text first.")

    st.header("")
    
# --- Stock Price Visualization Section ---
st.title("📈 Stock Price Visualization Dashboard")

# Sidebar inputs
ticker = st.sidebar.text_input("Enter Stock Ticker", "AAPL")
period = st.sidebar.selectbox("Select Period", ["1d", "5d", "1mo", "3mo", "6mo", "1y", "5y", "max"])
interval = st.sidebar.selectbox("Select Interval", ["1m", "5m", "15m", "1h", "1d", "1wk", "1mo"])

# Initialize session state
if "show_summarizer" not in st.session_state:
    st.session_state.show_summarizer = False

# Sidebar button to toggle summarizer
if st.sidebar.button("Open Text Summarizer"):
    st.session_state.show_summarizer = not st.session_state.show_summarizer

# --- Stock Data Visualization ---
data = yf.download(ticker, period=period, interval=interval)

if not data.empty:
    # Handle multi-index columns
    if isinstance(data.columns, pd.MultiIndex):
        data.columns = data.columns.droplevel(1)

    st.subheader(f"Stock Closing Price for {ticker}")
    fig = px.line(data, x=data.index, y="Close", title=f"{ticker} Closing Price")
    st.plotly_chart(fig)

    st.subheader("📊 Stock Data Table")
    st.dataframe(data.tail(20))
else:
    st.warning("⚠️ No data found. Try another ticker.")
